<?php
require_once 'wp_functions.php';

if (isset($_GET['process']) && $_GET['process'] === 'true') {
    // Prevent timeouts
    set_time_limit(0);
    ignore_user_abort(true);
    ini_set('max_execution_time', 0);
    
    header('Content-Type: text/event-stream');
    header('Cache-Control: no-cache');
    header('Connection: keep-alive');

    // Get data from GET parameters since we're using EventSource
    $siteUrl = $_GET['site_url'] ?? '';
    $username = $_GET['username'] ?? '';
    $password = $_GET['password'] ?? '';
    $jsonData = $_GET['jsonData'] ?? '';

    if (empty($siteUrl) || empty($username) || empty($password) || empty($jsonData)) {
        echo "event: error\n";
        echo "data: " . json_encode(["message" => "Tüm alanların doldurulması zorunludur"]) . "\n\n";
        exit();
    }

    // Validate and decode JSON data
    $articles = json_decode($jsonData, true);
    if ($articles === null) {
        echo "event: error\n";
        echo "data: " . json_encode(["message" => "Geçersiz JSON formatı: " . json_last_error_msg()]) . "\n\n";
        exit();
    }

    if (!is_array($articles)) {
        echo "event: error\n";
        echo "data: " . json_encode(["message" => "JSON verisi bir dizi içinde olmalıdır"]) . "\n\n";
        exit();
    }

    $total = count($articles);
    $current = 0;
    $results = array();
    $batchSize = 10; // Process 10 posts at a time

    // Process articles in batches
    $batches = array_chunk($articles, $batchSize);
    foreach ($batches as $batch) {
        foreach ($batch as $article) {
            $current++;
            
            // Send heartbeat to keep connection alive
            echo "event: heartbeat\n";
            echo "data: " . json_encode(['timestamp' => time()]) . "\n\n";
            ob_flush();
            flush();
            
            // Check if post exists first
            $existingPost = checkExistingPost($article['title'], $siteUrl, $username, $password);
            if ($existingPost !== null) {
                $result = array(
                    'index' => $current,
                    'title' => $article['title'],
                    'post_id' => $existingPost['id'],
                    'image_id' => null,
                    'status' => 'exists',
                    'message' => "Post already exists"
                );
                
                echo "event: progress\n";
                echo "data: " . json_encode([
                    'current' => $current,
                    'total' => $total,
                    'result' => $result
                ]) . "\n\n";
                ob_flush();
                flush();
                
                $results[] = $result;
                continue;
            }

            $imageId = null;
            
            // Upload image if exists
            if (!empty($article['main_image'])) {
                $imageId = uploadImageToWordPress($article['main_image'], $siteUrl, $username, $password);
            }
            
            // Create post
            $postResult = createWordPressPost($article, $imageId, $siteUrl, $username, $password);
            
            $result = array(
                'index' => $current,
                'title' => $article['title'],
                'post_id' => $postResult['post']['id'],
                'image_id' => $imageId,
                'status' => $postResult['status'],
                'message' => $postResult['message']
            );
            
            echo "event: progress\n";
            echo "data: " . json_encode([
                'current' => $current,
                'total' => $total,
                'result' => $result
            ]) . "\n\n";
            ob_flush();
            flush();
            
            $results[] = $result;
            
            // Small delay between posts to prevent overwhelming the server
            usleep(100000); // 100ms delay
        }
        
        // Delay between batches
        sleep(1);
    }

    echo "event: complete\n";
    echo "data: " . json_encode(['message' => 'All posts processed successfully']) . "\n\n";
    exit();
}
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>WordPress Post Creator</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        .poster-container {
            max-width: 800px;
            margin: 2rem auto;
            padding: 1rem;
        }
        #progressBar {
            transition: width 0.3s ease-in-out;
        }
        #logArea {
            max-height: 300px;
            overflow-y: auto;
            margin-bottom: 1rem;
        }
        .log-item {
            padding: 0.5rem;
            border-bottom: 1px solid #eee;
        }
        .log-item:last-child {
            border-bottom: none;
        }
    </style>
</head>
<body>
    <div class="poster-container">
        <h1 class="text-center mb-4">WordPress Post Creator</h1>
        
        <div id="progressContainer" class="mb-4" style="display: none;">
            <div class="progress">
                <div id="progressBar" class="progress-bar" role="progressbar" style="width: 0%"></div>
            </div>
            <div id="progressText" class="text-center mt-2"></div>
        </div>

        <div id="logArea" class="border rounded p-3 mb-4" style="display: none;">
            <!-- Logs will be displayed here -->
        </div>

        <form id="posterForm" method="POST" class="mb-4">
            <div class="mb-3">
                <label for="site_url" class="form-label">WordPress Site URL</label>
                <input type="url" class="form-control" id="site_url" name="site_url" required>
            </div>
            
            <div class="mb-3">
                <label for="username" class="form-label">Kullanıcı Adı</label>
                <input type="text" class="form-control" id="username" name="username" required>
            </div>
            
            <div class="mb-3">
                <label for="password" class="form-label">Şifre</label>
                <input type="password" class="form-control" id="password" name="password" required>
            </div>
            
            <div class="mb-3">
                <label for="jsonData" class="form-label">JSON Data</label>
                <textarea class="form-control" id="jsonData" name="jsonData" rows="10" required></textarea>
            </div>
            
            <button type="submit" class="btn btn-primary">Gönderileri Oluştur</button>
        </form>

        <div id="resultsTable" class="table-responsive" style="display: none;">
            <h3 class="mb-3">İşlem Sonuçları</h3>
            <table class="table table-striped">
                <thead>
                    <tr>
                        <th>#</th>
                        <th>Başlık</th>
                        <th>Post ID</th>
                        <th>Image ID</th>
                        <th>Durum</th>
                        <th>Mesaj</th>
                    </tr>
                </thead>
                <tbody id="resultsBody">
                </tbody>
            </table>
        </div>
    </div>

    <script>
        document.getElementById('posterForm').addEventListener('submit', function(e) {
            e.preventDefault();
            
            const progressContainer = document.getElementById('progressContainer');
            const progressBar = document.getElementById('progressBar');
            const progressText = document.getElementById('progressText');
            const logArea = document.getElementById('logArea');
            const resultsTable = document.getElementById('resultsTable');
            const resultsBody = document.getElementById('resultsBody');
            
            progressContainer.style.display = 'block';
            logArea.style.display = 'block';
            logArea.innerHTML = '';
            resultsTable.style.display = 'none'; // Initially hide results table
            resultsBody.innerHTML = '';
            
            // Create URL parameters with properly encoded data
            const params = new URLSearchParams();
            params.append('process', 'true');
            params.append('site_url', this.site_url.value);
            params.append('username', this.username.value);
            params.append('password', this.password.value);
            params.append('jsonData', this.jsonData.value);
            
            const evtSource = new EventSource('?' + params.toString());
            
            let results = [];
            
            evtSource.onmessage = function(event) {
                const data = JSON.parse(event.data);
                console.log(data);
            };
            
            evtSource.addEventListener('progress', function(event) {
                const data = JSON.parse(event.data);
                const percentage = (data.current / data.total) * 100;
                
                progressBar.style.width = percentage + '%';
                progressText.textContent = `İşleniyor: ${data.current} / ${data.total}`;
                
                // Add log entry
                const logItem = document.createElement('div');
                logItem.className = 'log-item';
                logItem.textContent = `${data.result.status === 'exists' ? 'Mevcut Post' : 'İşlendi'}: ${data.result.title}`;
                logArea.insertBefore(logItem, logArea.firstChild);
                
                // Store result
                results.push(data.result);
            });
            
            evtSource.addEventListener('error', function(event) {
                const data = JSON.parse(event.data);
                logArea.innerHTML = `<div class="alert alert-danger">${data.message}</div>` + logArea.innerHTML;
                evtSource.close();
            });
            
            evtSource.addEventListener('complete', function(event) {
                const data = JSON.parse(event.data);
                logArea.innerHTML = `<div class="alert alert-success">${data.message}</div>` + logArea.innerHTML;
                
                // Show and populate results table after completion
                resultsTable.style.display = 'block';
                results.forEach(result => {
                    const row = document.createElement('tr');
                    row.innerHTML = `
                        <td>${result.index}</td>
                        <td>${result.title}</td>
                        <td>${result.post_id}</td>
                        <td>${result.image_id || 'N/A'}</td>
                        <td>${result.status}</td>
                        <td>${result.message}</td>
                    `;
                    resultsBody.appendChild(row);
                });
                
                evtSource.close();
            });
            
            evtSource.addEventListener('heartbeat', function(event) {
                const data = JSON.parse(event.data);
                console.log(`Heartbeat received at ${data.timestamp}`);
            });
        });
    </script>
</body>
</html>
